let currentBoard, originalBoard, startTime, timerInterval;

function shuffle(arr) {
  for (let i = arr.length - 1; i > 0; i--) {
    const j = Math.floor(Math.random() * (i + 1));
    [arr[i], arr[j]] = [arr[j], arr[i]];
  }
  return arr;
}

function generateFullSudoku() {
  const board = Array.from({ length: 9 }, () => Array(9).fill(0));
  function isValid(b, r, c, n) {
    for (let x = 0; x < 9; x++) if (b[r][x] === n || b[x][c] === n) return false;
    const sr = r - r % 3, sc = c - c % 3;
    for (let i = 0; i < 3; i++) for (let j = 0; j < 3; j++)
      if (b[sr + i][sc + j] === n) return false;
    return true;
  }
  function fill(b) {
    for (let r = 0; r < 9; r++) {
      for (let c = 0; c < 9; c++) {
        if (b[r][c] === 0) {
          const nums = shuffle([1,2,3,4,5,6,7,8,9]);
          for (let n of nums) {
            if (isValid(b, r, c, n)) {
              b[r][c] = n;
              if (fill(b)) return true;
              b[r][c] = 0;
            }
          }
          return false;
        }
      }
    }
    return true;
  }
  fill(board);
  return board;
}

function removeCells(board, count = 45) {
  const b = board.map(row => row.slice());
  let removed = 0;
  while (removed < count) {
    const r = Math.floor(Math.random() * 9);
    const c = Math.floor(Math.random() * 9);
    if (b[r][c] !== '') {
      b[r][c] = '';
      removed++;
    }
  }
  return b;
}

function generateSudoku(difficulty = 45) {
  return removeCells(generateFullSudoku(), difficulty);
}

function drawBoard(board) {
  const sudoku = document.getElementById('sudoku');
  sudoku.innerHTML = '';
  for (let i = 0; i < 9; i++) {
    for (let j = 0; j < 9; j++) {
      const input = document.createElement('input');
      input.maxLength = 1;
      input.className = 'cell';
      if (originalBoard[i][j] !== '') {
        input.value = originalBoard[i][j];
        input.disabled = true;
        input.classList.add('prefilled');
      } else {
        input.value = board[i][j] || '';
        input.addEventListener('input', () => handleInput(i, j, input));
      }
      sudoku.appendChild(input);
    }
  }
}

function handleInput(row, col, el) {
  const val = parseInt(el.value);
  currentBoard[row][col] = isNaN(val) ? '' : val;
  saveToLocal();
  if (checkWin()) winGame();
}

function checkWin() {
  for (let i = 0; i < 9; i++) {
    for (let j = 0; j < 9; j++) {
      const val = currentBoard[i][j];
      if (!val || !checkValid(i, j, val)) return false;
    }
  }
  return true;
}

function checkValid(row, col, num) {
  for (let x = 0; x < 9; x++) {
    if (x !== col && currentBoard[row][x] == num) return false;
    if (x !== row && currentBoard[x][col] == num) return false;
  }
  const sr = row - row % 3, sc = col - col % 3;
  for (let i = 0; i < 3; i++) for (let j = 0; j < 3; j++) {
    const r = sr + i, c = sc + j;
    if ((r !== row || c !== col) && currentBoard[r][c] == num) return false;
  }
  return true;
}

function startTimer() {
  startTime = Date.now();
  timerInterval = setInterval(() => {
    const sec = Math.floor((Date.now() - startTime) / 1000);
    document.getElementById('timer').innerText = `⏱️ ${sec} giây`;
  }, 1000);
}

function stopTimer() {
  clearInterval(timerInterval);
}

function winGame() {
  stopTimer();
  localStorage.removeItem('sudoku_save');
  alert("🎉 Bạn đã hoàn thành Sudoku! Tuyệt vời!");
}

function saveToLocal() {
  const data = {
    currentBoard,
    originalBoard,
    startTime,
  };
  localStorage.setItem('sudoku_save', JSON.stringify(data));
}

function loadFromLocal() {
  const saved = localStorage.getItem('sudoku_save');
  if (!saved) return false;
  const data = JSON.parse(saved);
  currentBoard = data.currentBoard;
  originalBoard = data.originalBoard;
  startTime = data.startTime || Date.now();
  drawBoard(currentBoard);
  startTimer();
  document.getElementById('start-menu').style.display = 'none';
  document.getElementById('game-area').style.display = 'block';
  return true;
}

function startGame(forceNew = false) {
  stopTimer();
  if (!forceNew && loadFromLocal()) return;
  const diff = parseInt(document.getElementById('difficulty-select').value);
  currentBoard = generateSudoku(diff);
  originalBoard = JSON.parse(JSON.stringify(currentBoard));
  drawBoard(currentBoard);
  startTimer();
  document.getElementById('start-menu').style.display = 'none';
  document.getElementById('game-area').style.display = 'block';
  saveToLocal();
}

function resetGame() {
  currentBoard = JSON.parse(JSON.stringify(originalBoard));
  drawBoard(currentBoard);
  saveToLocal();
}

window.onload = () => {
  loadFromLocal();
};
